import json
import random
from pathlib import Path

import discord
import emoji
import requests

leagues = {}
autocomplete =[]

POWERS_OF_2 = [1,2,4,8,16,32,64,128,256,512,1024,2048,4096]

class MMOLBApiary:
	URL_BASE = "https://mmolb.com/api/"
	URL_NONAPI = "https://mmolb.com/"
	@staticmethod
	def get_time():
		try:
			resp = requests.get(f"{MMOLBApiary.URL_BASE}time")
			return resp.json()
		except requests.exceptions.RequestException as e:
			print(f"Failed to get time")
			print (e)
			return None

	@staticmethod
	def get_day():
		day = MMOLBApiary.get_time()
		if day is None:
			return -1
		else:
			return day["season_day"]

	@staticmethod
	def get_games_for_day(day):
		try:
			resp = requests.get(f"{MMOLBApiary.URL_BASE}day-games/{day}")
			if not resp.ok:
				print(f"Failed to get games for day {day}")
				return None
			return resp.json()
		except requests.exceptions.RequestException as e:
			print(f"Failed to get games for day {day}")
			print (e)
			return None

	@staticmethod
	def get_team_json(team_id):
		try:
			resp = requests.get(f"{MMOLBApiary.URL_BASE}team/{team_id}")
			if not resp.ok:
				print(f"Failed to get team {team_id}")
				return None
			return resp.json()
		except requests.exceptions.RequestException as e:
			print(f"Failed to get team {team_id}")
			print (e)
			return None
	@staticmethod
	def get_team_obj(team_id):
		try:
			resp = requests.get(f"{MMOLBApiary.URL_BASE}team/{team_id}")
			if not resp.ok:
				print(f"Failed to get team {team_id}")
				return None
			team_json = resp.json()
			return Team(team_json["_id"], team_json["Name"], team_json["Location"], team_json["Emoji"], team_json["Color"], team_json)
		except requests.exceptions.RequestException as e:
			print(f"Failed to get team {team_id}")
			print (e)
			return None
	@staticmethod
	def get_live_game_json_by_team(team):
		return MMOLBApiary.get_live_game_json_by_id(team.id)
	@staticmethod
	def get_live_game_json_by_id(team_id):
		try:
			resp = requests.get(f"{MMOLBApiary.URL_BASE}game-by-team/{team_id}")
			return resp.json()
		except requests.exceptions.RequestException as e:
			print(f"Failed to get game for: {team_id}")
	@staticmethod
	def get_game_json(game_id):
		try:
			resp = requests.get(f"{MMOLBApiary.URL_BASE}game/{game_id}")
			if resp.ok:
				return resp.json()
			else:
				print(resp.json())
				return None
		except requests.exceptions.RequestException as e:
			print(f"Failed to get game for: {game_id}")
			return None
	@staticmethod
	def get_spotlight_game_json():
		try:
			resp = requests.get(f"{MMOLBApiary.URL_BASE}spotlight")
			return resp.json()
		except requests.exceptions.RequestException as e:
			print(f"Failed to get spotlight game.")

	@staticmethod
	def get_lesser_league(league_id):
		try:
			resp = requests.get(f"{MMOLBApiary.URL_BASE}league/{league_id}")
			return resp.json()
		except requests.exceptions.RequestException as e:
			print(f"Failed to get league for: {league_id}")
			return None
	@staticmethod
	def get_state():
		try:
			resp = requests.get(f"{MMOLBApiary.URL_BASE}state")
			return resp.json()
		except requests.exceptions.RequestException as e:
			print(f"Failed to get state.")
			return None
	@staticmethod
	def get_team_schedule(team_id, season = None):
		seas = ""
		if season is not None:
			seas = f"?season={season}"
		try:
			resp = requests.get(f"{MMOLBApiary.URL_BASE}team-schedule/{team_id}{seas}")
			return resp.json()
		except requests.exceptions.RequestException as e:
			print(f"Failed to get team schedule for: {team_id}")
			return None
	@staticmethod
	def get_forecast():
		try:
			resp = requests.get(f"{MMOLBApiary.URL_BASE}weather-forecast")
			if not resp.ok:
				print(f"Failed to get weather forecast.")
				return None
			return resp.json()
		except requests.exceptions.RequestException as e:
			print(f"Failed to get weather forecast.")
			return None


class SeasonTime:
	def __init__(self):
		time_json = MMOLBApiary.get_time()
		if time_json is None:
			print("wuh oh")
			self.season_day = None
			self.season_number = None
			self.season_status = None
		else:
			self.season_day = time_json["season_day"]
			self.season_number = time_json["season_number"]
			self.season_status = time_json["season_status"]
	def __str__(self):
		return f"Season {self.season_number} Day {self.season_day} ({self.season_status})"
	def __eq__(self, other):
		return (self.season_number == other.season_number
				and self.season_status == other.season_status
				and self.season_day == other.season_day)
	def update_time(self):
		time_json = MMOLBApiary.get_time()
		if time_json is None:
			return False
		else:
			self.season_day = time_json["season_day"]
			self.season_number = time_json["season_number"]
			self.season_status = time_json["season_status"]
			return True

def get_subleague_gen(subleague_name):
	return (x for x in leagues if x.name.casefold() == subleague_name.casefold())

	#adds a subleague and saves the file
def add_subleague(subleague_name, subleague_emoji, subleague_channel):
	global autocomplete
	subleague = SubLeague(subleague_name, subleague_emoji, subleague_channel)
	leagues[subleague_name.lower()] = subleague
	serialize()
	autocomplete.append(subleague_name)
	autocomplete.sort()
	return subleague

	#adds a subleague *without* saving the file
def load_subleague(subleague_name, subleague_emoji, subleague_channel):
	subleague = SubLeague(subleague_name, subleague_emoji, subleague_channel)
	leagues[subleague_name.lower()] = subleague
	return subleague

def serialize():
	leagues_ser = []
	for subleague in leagues.values():
		subleague_ser = {"name": subleague.name, "emoji": subleague.emoji, "channel": subleague.channel, "teams": []}
		for team in subleague.teams:
			subleague_ser["teams"].append(team)
		leagues_ser.append(subleague_ser)
	open("leagues.json", "w").write(json.dumps(leagues_ser, indent=4))

def unserialize():
	print("Unserializing. Don't.")
	global autocomplete
	autocomplete = []
	try:
		with open("leagues.json") as json_file:
			data = json.load(json_file)
			for subleague in data:
				sl_obj = load_subleague(subleague["name"], subleague["emoji"], subleague["channel"])
				for team_id in subleague["teams"]:
					#this makes me feel bad
					sl_obj.load_team(MMOLBApiary.get_team_obj(team_id))
				autocomplete.append(subleague["name"])
			autocomplete.sort()

	except IOError as e:
		print("aint there")
		#uhhh
		global leagues
		leagues = {}
	print("Okay we're gucci now")


class SubLeague:
	def __init__(self, league_name, league_emoji, channel):
		self.name = league_name
		self.emoji = league_emoji
		self.teams = {}
		self.quick_teams = set()
		self.channel = channel
		#self.watchers = []
	def add_team(self, team):
		self.teams[team.id] = team
		self.quick_teams.add(team.id)
		serialize()
	def load_team(self, team):
		self.teams[team.id] = team
		self.quick_teams.add(team.id)
	def print_teams(self):
		return '\n'.join(str(x) for x in self.teams.values())
	def standings(self):
		return sorted(self.teams.values(), key=lambda team: (team.winpct(), team.rundiff()), reverse=True)

	def __str__(self):
		return f'{self.emoji} {self.name} League'
	def __repr__(self):
		return f'{self.emoji} {self.name} League'


class Team:
	def __init__(self, team_id, team_name, team_location, team_emoji, color, json):
		self.name = team_name
		self.location = team_location
		self.emoji = team_emoji
		self.color = color
		self.id = team_id
		self.json = json
		if self.json:
			self.players =  self.json["Players"]
		else:
			self.players = None

	def update(self):
		self.json = MMOLBApiary.get_team_json(self.id)
		if self.json is None:
			return False
		self.name = self.json['Name']
		self.location = self.json['Location']
		self.emoji = self.json['Emoji']
		self.color = self.json['Color']
		self.players = self.json['Players']
		return True
	def pretty_name(self):
		return f'{self.emoji} {self.location} {self.name}'
	def wins(self): return self.json["Record"]["Regular Season"]["Wins"]
	def losses(self): return self.json["Record"]["Regular Season"]["Losses"]
	def rundiff(self):
		return self.json["Record"]["Regular Season"]["RunDifferential"]
	def winpct(self):
		if self.wins() + self.losses() < 1:
			return 0
		else:
			return round(self.wins() / (self.wins() + self.losses()), 3)
	def savefile(self, seatime: SeasonTime):
		Path(f"SeasonStats/{seatime.season_number}").mkdir(parents=True, exist_ok=True)
		with open(f"SeasonStats/{seatime.season_number}/{self.id} - {seatime.season_day}.json","w") as f:
			json.dump(self.json, f, indent="\t")

	def __str__(self):
		return f'{self.emoji} {self.location} {self.name}'
	def __repr__(self):
		return f'{self.emoji} {self.location} {self.name}'

class Game:
	URL_BASE = "https://mmolb.com/watch/"
	def __init__(self, game_id, away_name, away_id, away_sp, home_name, home_id, home_sp, weather_dict, color, home_color):
		self.game_id = game_id
		self.away_name = away_name
		self.away_id = away_id
		self.away_json = MMOLBApiary.get_team_json(away_id)
		self.away_sp_slot = away_sp
		self.away_sp_name = self.get_pitcher_name(self.away_json, int(away_sp[-1]))
		self.home_name = home_name
		self.home_id = home_id
		self.home_json = MMOLBApiary.get_team_json(home_id)
		self.home_sp_name = self.get_pitcher_name(self.home_json, int(home_sp[-1]))
		self.home_sp_slot = home_sp
		self.ballpark =  self.home_json["BallparkName"]
		self.weather_dict = weather_dict
		self.color = color
		self.home_color = home_color
		#extras
		self.leaguestr = None
		self.showdown = False
		self.spotlight = False
		self.watch = True
	@staticmethod
	def get_pitcher_name(teamjson, sp_num):
		playerdict = teamjson["Players"][sp_num + 8]
		return (f"{playerdict['Emoji']} [{playerdict['FirstName']} {playerdict['LastName']}]({MMOLBApiary.URL_NONAPI}player/{playerdict['PlayerID']}) "
				f"({playerdict['Stats'].get('wins', 0)}-{playerdict['Stats'].get('losses', 0)})")

	def inter_subleague(self) -> str|None:
		away = None
		home = None
		for sl in leagues.values():
			if self.away_id in sl.quick_teams:
				away = sl.emoji
			if self.home_id in sl.quick_teams:
				home = sl.emoji
		if away != home and away is not None and home is not None:
			return f"{away} vs {home}"
		return None

	def get_withered_alt(self):
		return {"away": self.check_wither(self.away_json), "home": self.check_wither(self.home_json)}

	def check_wither(self, team_json):
		wither_list = []
		if len(team_json["Modifications"]) > 0:
			for i in team_json["Modifications"]:
				if i["Name"] == "Corrupted":
					team_url = f"{MMOLBApiary.URL_NONAPI}/team/{team_json['_id']}"
					positions = ["C", "1B", "2B", "3B", "SS", "LF", "CF", "RF", "RP1", "RP2", "RP3", "CL", "5B", "LS", "XF"]
					random.shuffle(positions)
					stinky = [f"[{team_json["Emoji"]} {positions[0]} um."
							  f"\n{team_json["Emoji"]} {positions[1]} all of them."
							  f"\n{team_json["Emoji"]} {positions[2]} good luck?]({team_url})",

							  f"[{team_json["Emoji"]} {positions[0]} Hazards Detected."
							  f"\n{team_json["Emoji"]} {positions[1]} Hazards Detected."
							  f"\n{team_json["Emoji"]} {positions[2]} Hazards Detected."
							  f"\n{team_json["Emoji"]} {positions[3]} Hazards Detected."
							  f"\n{team_json["Emoji"]} {positions[4]} Hazards Detected."
							  f"\n{team_json["Emoji"]} {positions[5]} Hazards Detec-]({team_url})",
							  f"[{team_json["Emoji"]} {positions[0]} why doesn't the commissioner"
							  f"\n{team_json["Emoji"]} {positions[1]} just ban these teams?"
							  f"\n{team_json["Emoji"]} {positions[2]} i mean."
							  f"\n{team_json["Emoji"]} {positions[3]} how much is the league"
							  f"\n{team_json["Emoji"]} {positions[4]} spending on purifications...]({team_url})",]
					return [random.choice(stinky)]
		plist = team_json["Players"]
		length = 0
		for player in plist:
			for mod in player["Modifications"]:
				if mod["Name"] == "Corrupted":
					if (player["Position"] == "SP" and not
					(player["Slot"] == self.away_sp_slot or player["Slot"] == self.home_sp_slot)):
						break
					wither_string = f"{player['Emoji']} [{player["Slot"]} {player['FirstName']} {player['LastName']}]({MMOLBApiary.URL_NONAPI}player/{player['PlayerID']})"
					length += len(wither_string) + 1
					#todo: check to see if between 940 and 1024
					#maybe fix the +1
					if length > 940:
						wither_list.append(f"{team_json["Emoji"]} [there's too many...]({MMOLBApiary.URL_NONAPI}/team/{team_json['_id']})")
						return wither_list
					wither_list.append(wither_string)
					break
		return wither_list

	def generate_embed(self):
		desc_list = []
		if self.weather_dict['Emoji'] == emoji.emojize(":partying_face:"):
			self.weather_dict['Emoji'] = emoji.emojize(":party_popper:")
			#its my bot i get to change the emoji
		if self.spotlight:
			desc_list.append("**👁️✍️ This game is being Observed.**")
		if self.showdown:
			desc_list.append(f"**{self.leaguestr} Showdown**")
			self.color = self.home_color
		else:
			il = self.inter_subleague()
			if il is not None:
				desc_list.append(f"**{self.inter_subleague()}**")

		embed = discord.Embed(title=f"{self.away_name} @ {self.home_name}",
							  url=f"{Game.URL_BASE}{self.game_id}",
							  description='\n'.join(desc_list),
							  color=discord.Color.from_str("#"+self.color))
		embed.set_footer(text=self.ballpark)
		embed.add_field(name="Starting Pitchers", value=f"{self.away_sp_name}\n {self.home_sp_name}", inline=True)
		embed.add_field(name=f"Weather: {self.weather_dict['Name']}",
						value=f"{self.weather_dict['Tooltip']}\n{self.weather_dict['Emoji']}\n<:blobok:1389808568290770984>", inline=True)

		if self.weather_dict['Name'] == "Wither":
			#stinky
			#currently not needed but keeping for the time being.
			wd = self.get_withered_alt()
			if len(wd["away"]) + len(wd["home"]) > 0:
				with_away = "\n".join(wd["away"])
				with_home = "\n".join(wd["home"])
				print(len(with_away), len(with_home))
				if len(with_away) + len(with_home) > 1024:
					embed.add_field(name="", value="", inline=True)
					embed.add_field(name="Hazards Detected (Away)",value=with_away, inline=True)
					embed.add_field(name="Hazards Detected (Home)",value=with_home, inline=True)
					embed.add_field(name="", value=" ", inline=True)
				else:
					embed.add_field(name="Hazards Detected", value=with_away + "\n" + with_home, inline=False)
		return embed

	def __eq__(self, other):
		return self.game_id == other.game_id
	def __str__(self):
		return (f'{self.game_id} {self.away_name} {self.away_id} {self.away_sp_name} '
				f'{self.home_id} {self.home_name} {self.home_sp_name}')

class Forecast:
	def __init__(self):
		self.weather_list = []
		weather_json = MMOLBApiary.get_forecast()
		if weather_json is not None:
			self.season_number = weather_json["season_number"]
			for i in weather_json["forecast"]:
				self.weather_list.append(i)

	def generate_embed(self):
		out = []
		for i in self.weather_list:
			if i['emoji'] == emoji.emojize(":partying_face:"):
				i['emoji'] = emoji.emojize(":party_popper:")
			if i["probability"] >= 10:
				out.append(f"**{i['emoji']} {i["name"]} - {i['probability']}%**")
			else:
				out.append(f"{i['emoji']} {i["name"]} - {i['probability']}%")
		embed = discord.Embed(title=f"Season {self.season_number} Weather Outlook:",description="\n".join(out),
							  color=discord.Color.from_str("#75d5ff"))
		embed.set_footer(text=f"i heard they said it looked like {random.choice(self.weather_list)['emoji']}")
		return embed


if __name__ == "__main__":
	fo = Forecast()