import json
import random

from enum import Enum

class Formatting(Enum):
	PERCENT = "PER"
	NUMBER = "NUM"

class Ops(Enum):
	ADD = "ADD"
	SUB = "SUB"
	MUL = "MUL"
	DIV = "DIV"

class StatType(Enum):
	BATTING = "BATTING"
	PITCHING = "PITCHING"
	DEFENSE = "DEFENSE"
	BASERUNNING = "BASERUNNING"

def calculate(inp: list|int|str, playerstats):
	output = 0
	if type(inp) == int:
		#number, we're all good
		return inp
	elif type(inp) == str:
		#getting stat from json
		if inp in playerstats:
			print (f"{inp}: {playerstats[inp]}")
			return playerstats[inp]
		else:
			return 0
	elif type(inp) == list:
		#need to do an operation
		if inp[0] == Ops.ADD:
			for i in inp[1:]:
				output += calculate(i,playerstats)
		elif inp[0] == Ops.SUB:
			output = calculate(inp[1],playerstats)
			for i in inp[2:]:
				output -= calculate(i,playerstats)
		elif inp[0] == Ops.MUL:
			output = 1
			for i in inp[1:]:
				output *= calculate(i,playerstats)
		elif inp[0] == Ops.DIV:
			if len(inp) > 3:
				print("Only works on simple div, sorr")
				return 0
			numerator = calculate(inp[1],playerstats)
			denominator = calculate(inp[2],playerstats)
			#solving div/0
			output = numerator / denominator if denominator else float("inf")
		#print(f"{inp} -> {output}")
		return output
	else:
		print("I don't like what you've done.")
		return 0

class Statistic:
	def __init__(self, name, formatting: Formatting, positive: bool, formula, qualifier: tuple | None = None, precision: float = None):
		self.name = name
		self.formatting = formatting
		self.positive = positive
		self.formula = formula
		self.qualifier = qualifier
		if precision is not None:
			self.precision = precision
		else:
			self.precision = 0.2
	def calc_leaderboard(self, players):
		output = []
		random.shuffle(players)
		for player in players:
			plname = f"{player['FirstName']} {player['LastName']}"
			stats = player["Stats"]
			if self.qualifier:
				if calculate(self.qualifier[0], stats) < self.qualifier[1]:
					continue
			calc = calculate(self.formula, stats)
			print(f"{plname}: {calc}")
			if self.formatting == Formatting.PERCENT:
				output.append({"name": plname,"emoji":player['Emoji'], "sort": calc, "display": f"{calc:{self.precision}f}", "id":player['PlayerID']})
			else:
				output.append({"name": plname,"emoji":player['Emoji'], "sort": calc, "display": calc, "id":player['PlayerID']})
		output.sort(key=lambda t: t["sort"], reverse=self.positive)
		return output

# MF Statz Zone™
#TODO:
# figure out a way to do the Worst players other than qualifying
# rn those with ("plate_appearances", 1), ("outs", 1), ([Ops.ADD, "putouts", "assists"], 1), and ([Ops.ADD, "runners_caught_stealing", "allowed_stolen_bases"], 1)
# are in the workaround zone
# sort by importance
# add random option

#TODO:
# negative counting stats are currently flipped - consider changing or reimplementing Best
# stats to add:
# balks
# strikeouts w/risp (krisp)

Statistics = {
	# Batting
	## Website Stats
	"AVG":Statistic("Batting Average", Formatting.PERCENT, True,[Ops.DIV, [Ops.ADD, "singles", "doubles", "triples", "home_runs"], "at_bats"], ("plate_appearances", 10), precision=0.3),
	#"OBP":Statistic("On-Base Percentage", Formatting.PERCENT, True, [Ops.DIV,[Ops.ADD, "singles", "doubles", "triples", "home_runs","walked","hit_by_pitch"],[Ops.ADD, "at_bats","walked","hit_by_pitch","sac_flies","sacrifice_double_plays"]],("at_bats", 10)),
	"OBP":Statistic("On-Base Percentage", Formatting.PERCENT,True,[Ops.DIV,[Ops.ADD, "singles", "doubles", "triples", "home_runs","walked","hit_by_pitch"],"plate_appearances"],("plate_appearances", 10), precision=0.3),
	"SLG":Statistic("Slugging Percentage",Formatting.PERCENT,True,[Ops.DIV,[Ops.ADD,"singles",[Ops.MUL,2,"doubles"],[Ops.MUL,3,"triples"],[Ops.MUL,4,"home_runs"]],"at_bats"],("plate_appearances", 10), precision=0.3),
	"OPS":Statistic("On-Base plus Slugging", Formatting.PERCENT, True, [Ops.ADD, [Ops.DIV,[Ops.ADD, "singles", "doubles", "triples", "home_runs","walked","hit_by_pitch"],"plate_appearances"], [Ops.DIV,[Ops.ADD,"singles",[Ops.MUL,2,"doubles"],[Ops.MUL,3,"triples"],[Ops.MUL,4,"home_runs"]],"at_bats"]], ("plate_appearances", 10), precision=0.3),
	"H":Statistic("Hits",Formatting.NUMBER,True,[Ops.ADD, "singles", "doubles", "triples", "home_runs"], ("plate_appearances", 1)),
	"1B":Statistic("Singles",Formatting.NUMBER,True,"singles", ("plate_appearances", 1)),
	"2B":Statistic("Doubles",Formatting.NUMBER,True,"doubles", ("plate_appearances", 1)),
	"3B":Statistic("Triples",Formatting.NUMBER,True,"triples", ("plate_appearances", 1)),
	"HR":Statistic("Home Runs",Formatting.NUMBER,True,"home_runs", ("plate_appearances", 1)),
	"BB":Statistic("Walks",Formatting.NUMBER,True,"walked", ("plate_appearances", 1)),
	#technically not on the website but cmon
	"HBP":Statistic("Hit By Pitch", Formatting.NUMBER, True, "hit_by_pitch", ("plate_appearances", 1)),
	"SB":Statistic("Stolen Bases", Formatting.NUMBER, True, "stolen_bases", ("plate_appearances", 1)),
	"CS":Statistic("Caught Stealing", Formatting.NUMBER, True, "caught_stealing",([Ops.ADD, "stolen_bases", "caught_stealing"], 1)),
	"GIDP":Statistic("Grounded Into Double Plays", Formatting.NUMBER, True, "grounded_into_double_play", ("plate_appearances", 10)),
	## Bee Stats™
	"SB%":Statistic("Stolen Base Percentage", Formatting.PERCENT, True, [Ops.DIV, "stolen_bases",[Ops.ADD, "stolen_bases", "caught_stealing"]], ([Ops.ADD, "stolen_bases", "caught_stealing"], 1)),
	"RBI":Statistic("Runs Batted In", Formatting.NUMBER, True, "runs_batted_in", ("plate_appearances", 1)),
	"TB":Statistic("Total Bases", Formatting.NUMBER, True,[Ops.ADD, "singles", [Ops.MUL, 2, "doubles"], [Ops.MUL, 3, "triples"], [Ops.MUL, 4, "home_runs"]], ("plate_appearances", 1)),
	"XBH": Statistic("Extra Base Hits", Formatting.NUMBER, True, [Ops.ADD, "doubles", "triples", "home_runs"], ("plate_appearances", 1)),
	"CLUTCH":Statistic("Clutch Percentage", Formatting.PERCENT, True,[Ops.DIV,[Ops.ADD,"singles_risp","doubles_risp","triples_risp","home_runs_risp","walked_risp","hit_by_pitch_risp"],"plate_appearances_risp"],("plate_appearances_risp", 1)),
	#"CS/HR": Statistic("Caught Stealing per Home Run", Formatting.PERCENT, False,[Ops.DIV, "caught_stealing", "home_runs"], ("plate_appearances", 10)),
	"BABIP":Statistic("Batting Average on Balls In Play", Formatting.PERCENT, True, [Ops.DIV,[Ops.ADD,"singles", "doubles", "triples"],[Ops.ADD,[Ops.SUB, "at_bats","struck_out","home_runs"], "sac_flies", "sacrifice_double_plays"]], ("plate_appearances", 10), precision=0.3),
	#Pitching
	## Website Stats
	"ERA":Statistic("Earned Run Average",Formatting.PERCENT, False,[Ops.MUL,9,[Ops.DIV,"earned_runs",[Ops.DIV,"outs",3]]],("outs",27)),
	"WHIP":Statistic("Walks plus Hits per Inning Pitched",Formatting.PERCENT,False,[Ops.DIV,[Ops.ADD,"walks","hit_batters","hits_allowed"],[Ops.DIV,"outs",3]],("outs",27)),
	#"WBHIP": Statistic("Walks, HBP, and Hits per Inning Pitched", Formatting.PERCENT, False,[Ops.DIV, [Ops.ADD, "walks","hit_batters", "hits_allowed"], [Ops.DIV, "outs", 3]], ("outs", 27)),
	"K/BB": Statistic("Strikeout to Walk Ratio",Formatting.PERCENT,True,[Ops.DIV,"strikeouts","walks"],("outs",27)),
	"K/9": Statistic("Strikeouts per 9 Innings",Formatting.PERCENT,True,[Ops.DIV,"strikeouts",[Ops.DIV,"outs",27]],("outs",27)),
	"H/9": Statistic("Hits per 9 Innings", Formatting.PERCENT, False,[Ops.DIV, "hits_allowed", [Ops.DIV, "outs", 27]], ("outs", 27)),
	"BB/9": Statistic("Walks per 9 Innings", Formatting.PERCENT, False,[Ops.DIV, "walks", [Ops.DIV, "outs", 27]], ("outs", 27)),
	"HR/9": Statistic("Home Runs per 9 Innings", Formatting.PERCENT, False,[Ops.DIV, "home_runs_allowed", [Ops.DIV, "outs", 27]], ("outs", 27)),
	"K":Statistic("Strikeouts", Formatting.NUMBER,True,"strikeouts", ("outs", 1)),
	"IP":Statistic("Innings Pitched", Formatting.PERCENT, True, [Ops.DIV,"outs", 3], ("outs", 27)),
	"BBP":Statistic("Walks Issued", Formatting.NUMBER,True,"walks", ("outs", 27)),
	"HA":Statistic("Hits Allowed", Formatting.NUMBER,True,"hits_allowed", ("outs", 27)),
	"HB":Statistic("Hit Batters", Formatting.NUMBER,True,"hit_batters", ("outs", 27)),
	"W":Statistic("Wins", Formatting.NUMBER, True,"wins", ("outs", 1)),
	"L":Statistic("Losses", Formatting.NUMBER, True,"losses", ("outs", 27)),
	"QS":Statistic("Quality Starts", Formatting.NUMBER,True,"quality_starts", ("outs", 1)),
	"SV":Statistic("Saves", Formatting.NUMBER,True,"saves", ([Ops.ADD, "saves", "blown_saves"], 5)),
	"BS":Statistic("Blown Saves", Formatting.NUMBER,True,"blown_saves", ([Ops.ADD, "saves", "blown_saves"], 1)),
	"G":Statistic("Games Pitched", Formatting.NUMBER,True,"appearances", ("outs", 1)),
	"GF":Statistic("Games Finished", Formatting.NUMBER, True, "games_finished", ("outs", 1)),
	"CG":Statistic("Complete Games", Formatting.NUMBER,True,"complete_games", ("outs", 1)),
	"SHO":Statistic("Shutouts", Formatting.NUMBER,True,"shutouts", ("outs", 1)),
	#"NH":Statistic("No Hitters", Formatting.NUMBER,True,"no_hitters", ("outs", 1)), #probably.
	## Bee Stats
	"SV%":Statistic("Save Percentage", Formatting.PERCENT, True, [Ops.DIV,"saves",[Ops.ADD, "saves", "blown_saves"]], ([Ops.ADD, "saves", "blown_saves"], 5)),
	"HB/9":Statistic("Hit Batters per 9 Innings",Formatting.PERCENT, False,[Ops.DIV,"hit_batters", [Ops.DIV,"outs", 27]], ("outs", 27)),
	"WHB/9":Statistic("Walks plus Hit Batters per 9 Innings", Formatting.PERCENT, False,[Ops.DIV,[Ops.ADD, "walks","hit_batters"],[Ops.DIV,"outs", 27]], ("outs", 27)),
	"MV":Statistic("🚶", Formatting.NUMBER,True,"mound_visits", ("outs", 1)),
	"PT":Statistic("Pitches Thrown", Formatting.NUMBER,True,"pitches_thrown", ("outs", 1)),
	"PT/A":Statistic("Pitches Thrown per Appearance", Formatting.PERCENT, True, [Ops.DIV, "pitches_thrown", "appearances"],("appearances", 1)),
	"PT/PA":Statistic("Pitches Thrown per Plate Appearance",Formatting.PERCENT,False,[Ops.DIV, "pitches_thrown", "batters_faced"],("outs", 9)),
	"BK":Statistic("Balks", Formatting.NUMBER,True,"balks", ("balks", 1)),
	# Defence
	## Website Stats
	"E":Statistic("Errors", Formatting.NUMBER, True,"errors", ([Ops.ADD, "putouts", "assists"], 10)),
	"A":Statistic("Assists", Formatting.NUMBER, True,"assists", ([Ops.ADD, "putouts", "assists"], 1)),
	"PO":Statistic("Putouts", Formatting.NUMBER, True,"putouts", ([Ops.ADD, "putouts", "assists"], 1)),
	"DP":Statistic("Double Plays", Formatting.NUMBER, True,"double_plays", ([Ops.ADD, "putouts", "assists"], 1)),
	"RCS":Statistic("Runners Caught Stealing", Formatting.NUMBER, True,"runners_caught_stealing", ([Ops.ADD, "runners_caught_stealing", "allowed_stolen_bases"], 1)),
	"RCS%":Statistic("Runners Caught Stealing Percentage", Formatting.PERCENT, True,[Ops.DIV, "runners_caught_stealing",[Ops.ADD, "runners_caught_stealing", "allowed_stolen_bases"]], ([Ops.ADD, "runners_caught_stealing", "allowed_stolen_bases"], 1)),
	# Extras I/g?
	"EJ":Statistic("Ejections", Formatting.NUMBER, True, "ejected")
}

autocomplete = list(Statistics.keys())


#testing
if __name__ == "__main__":
	with (open("test data/Just Match Fixers.json")) as json_file:
		data = json.load(json_file)
		json_file.close()
	players = data["Players"]
	with (open("test data/Women Absence Feeling.json")) as json_file:
		data = json.load(json_file)
		json_file.close()
	players += data["Players"]
	with (open("test data/Territory.json")) as json_file:
		data = json.load(json_file)
		json_file.close()
	players += data["Players"]
	for i in Statistics["HB/9"].calc_leaderboard(players):
		print(f"{i["emoji"]} {i['name']}: {i['display']}")
# data.unserialize()
# players = []
# for team in data.leagues["deeper"].teams.values():
# 	players += team.players
# stats = Statistics["HR/9"].calc_leaderboard(players)
# for i in stats:
# 	print(f"{i[0]}: {i[1]:0.3f}")
# for i in stats[:-4:-1]:
# 	print(f"{stats.index(i)}) {i[0]}: {i[1]:0.3f}")
# stats = []
# for player in players:
# 	stats.append((f"{player['Emoji']} {player['FirstName']} {player['LastName']}", calculate(BA, player['Stats'])))
# stats.sort(key=lambda f:f[1], reverse=True)
# print("Batting Avg. Leaders: ")
# for i in stats:
# 	print(f"{i[0]}: {i[1]:0.3f}")


