import io
import json
import random
from pathlib import Path

import discord
import matplotlib


import data
import color_contrast
from datetime import datetime
from matplotlib import pyplot as plt
from enum import Enum
from color_contrast import AccessibilityLevel

matplotlib.use('Agg')

# currently only sploon 1 unranked
splatoon_colors = [("#C83D79","#409D3B"),("#DA3781","#ed9408"),("#C93457","#048188"),
				   ("#cf581b","#141494"),("#799516","#6E068A"), ("#20837D","#DF641A"),
				   ("#20837D", "#DF641A"), ("#228CFF","#E85407"), ("#007EDC", "#E1A307"),
				   ("#26229F","#91B00B")]

def jsonKeys2int(x):
	if isinstance(x, dict):
		out = {}
		for k, v in x.items():
			if (k == 'true'):
				out[True] = v
			elif (k == 'false'):
				out[False] = v
			elif k.lstrip('-').isdigit():
				out[int(k)]=v
			else:
				out[k]=v
		return out
	return x

#palettes: unedited, +contrast (maybe check if needed?), high_contrast
#splatoon, mesmerizer - hidden :3
class Palette(Enum):
	UNEDITED = "UNEDITED"
	CONTRAST = "CONTRAST"
	HIGH_CONTRAST = "HIGH_CONTRAST"

	SPLATOON = "SPLATOON"
	MIKUTETO = "MIKUTETO"

	@classmethod
	def _missing_(cls, value):
		value = value.upper()
		for member in cls:
			if member.value == value:
				return member
		return Palette.CONTRAST



class Event:
	def __init__(self, inning, homeatbat, away_score, home_score, outs, on_1b = False, on_2b = False, on_3b = False, homeWin = False, message = None, url = None):
		self.inning = inning
		self.homeatbat = homeatbat == 1
		self.away_score = away_score
		self.home_score = home_score
		self.homelead = home_score - away_score
		self.outs = outs
		self.runners = (on_1b * 1) + (on_2b * 2) + (on_3b * 4)
		self.homeWin = homeWin
		self.message = message
		self.url = url

	def __str__(self):
		return (f"{"Bot" if self.homeatbat else "Top"} {self.inning} "
				f"Score: {self.away_score} - {self.home_score} "
				f"Outs: {self.outs} "
				f"Runners: {self.runners:03b} "
				f"Message: {self.message}")
	def __repr__(self):
		return self.__str__()
	def __eq__(self, other):
		if not isinstance(other, Event):
			return False
		if (self.inning != other.inning
				or self.homeatbat != other.homeatbat
				or self.homelead != other.homelead
				or self.outs != other.outs
				or self.runners != other.runners):
			return False
		else:
			return True

class EventWP:
	def __init__(self, event: Event, prev_event, wp, wpa):
		self.event = event
		self.prev_event = prev_event
		self.wp = wp
		self.wpa = wpa
	def __str__(self):
		return f"{self.event} Win Pct: {self.wp*100:0.2f} Win Pct Added: {self.wpa*100:0.2f}"
	def __repr__(self):
		return self.__str__()

	def pretty_prev_event(self, away_str: str, home_str: str) -> str:
		prev = self.prev_event.event
		return (f"{away_str} {prev.away_score} - {prev.home_score} {home_str} - "
				f"{"Bot" if prev.homeatbat else "Top"} {prev.inning}, "
				f"{prev.outs} {"Out" if  prev.outs == 1 else "Outs"}")
		# return (f"{"Bot" if prev.homeatbat else "Top"} {prev.inning} "
		# 		f"Score: {prev.away_score} - {prev.home_score} "
		# 		f"{runners}, {prev.outs} {"Out" if  prev.outs == 1 else "Outs"}")

	def pretty_runners(self):
		return f"https://images.swarm.computer/mmolbee/bases/{self.prev_event.event.runners:03b}.png"

	def pretty_message(self) -> str:
		return self.event.message.replace("<strong>", "*").replace("</strong>", "*")


# Runners on base:
## First = 1 Second = 2 Third = 4
#[inning][homeAtBat][homeLead][outs][runners] = {"homeWins": 0, "occurrences": 0, "homeWinPct": 0.0}

class WinPct:
	def __init__(self, imp: dict = None):
		self.table = {}
		if imp is not None and type(imp) == dict:
			self.table = imp
			return
		count = 0
		homeAtBatList = [False, True]

		#for inning in range(1, 11):
		for inning in range(1,11):
			self.table[inning] = {}
			for homeAtBat in homeAtBatList:
				self.table[inning][homeAtBat] = {}
				for homeLead in range(-7, 8):
					self.table[inning][homeAtBat][homeLead] = {}
					for outs in range(0, 3):
						self.table[inning][homeAtBat][homeLead][outs] = {}
						for runners in range(0, 8):
							self.table[inning][homeAtBat][homeLead][outs][runners] = {"homeWins": 0, "occurrences": 0, "homeWinPct": 0.0}
							if inning >= 9 and homeAtBat and homeLead > 0:
								self.table[inning][homeAtBat][homeLead][outs][runners] = {"homeWins": 1, "occurrences": 1, "homeWinPct": 1.0}
							count += 1
		print(count)
		self.table["awayWins"] = {"homeWins": 0, "occurrences": 1, "homeWinPct": 0.0}

	def insert_occurrence(self,inning,homeatbat,homelead,outs,runners,homewin):
		if outs == 3 and inning >=9 and homeatbat and homelead < 0:
			#home lost
			return
		if inning > 9:
			inning = 10
		if homelead > 7:
			homelead = 7
		if homelead < -7:
			homelead = -7
		occurrence = self.table[inning][homeatbat][homelead][outs][runners]
		if homewin:
			occurrence["homeWins"] += 1
		occurrence["occurrences"] += 1
		occurrence["homeWinPct"] = occurrence["homeWins"] / occurrence["occurrences"]

	def insert_occurrence_event(self, event: Event):
		self.insert_occurrence(event.inning, event.homeatbat, event.homelead, event.outs, event.runners, event.homeWin)

	def get_win_info(self, inning, homeatbat, homelead, outs, runners):
		if inning >= 9 and homeatbat and homelead < 0 and outs == 3:
			return self.table["awayWins"]
		if inning > 9:
			inning = 10
		if homelead > 7:
			homelead = 7
		if homelead < -7:
			homelead = -7
		return self.table[inning][homeatbat][homelead][outs][runners]
	def get_win_info_event(self, event: Event):
		return self.get_win_info(inning=event.inning, homeatbat=event.homeatbat, homelead=event.homelead, outs=event.outs, runners=event.runners)

	def get_win_pct(self, inning, homeatbat, homelead, outs, runners):
		return self.get_win_info(inning, homeatbat, homelead, outs, runners)["homeWinPct"]
	def get_win_pct_event(self, event: Event):
		return self.get_win_pct(inning=event.inning, homeatbat=event.homeatbat, homelead=event.homelead, outs=event.outs, runners=event.runners)

class GameParser:
	skips = ["PitchingMatchup","AwayLineup","HomeLineup","PlayBall","LiveNow","InningStart","InningEnd",]
	def __init__(self, game_id: str = None, local_file: str = None, local_json: dict = None):
		self.game_json = None
		self.event_list = []
		self.game_id = game_id
		if local_file is not None:
			with open(local_file, "r") as f:
				self.game_json = json.load(f)
		elif local_json is not None:
			self.game_json = local_json
		elif game_id is not None:
			self.game_json = data.MMOLBApiary.get_game_json(game_id)

		if self.game_json is None:
			print("sure whatever")
			return
		final_event = self.game_json["EventLog"][-1]
		self.home_win = final_event["home_score"] > final_event["away_score"]
		self.manfred = "Postseason" not in self.game_json["SeasonStatus"]
	def parse(self):
		if self.game_json is None or self.game_json["State"] != "Complete":
			print(f"Game {data.MMOLBApiary.URL_NONAPI}watch/{self.game_id} either doesn't exist or is not complete.")
			# todo: only for chunching
			#return []

		event_log = self.game_json["EventLog"]
		prev_event = None
		count = 0
		output = []
		for idx in event_log:
			if idx["event"] == 'GameOver':
				break
			if idx["event"] in self.skips:
				continue
			if idx["outs"] is None:
				idx["outs"] = 0
				if idx["inning"] < 9:
					if idx["inning_side"] == 0:
						idx["inning_side"] = 1
					else:
						idx["inning"] += 1
						idx["inning_side"] = 0
				else:
					if idx["inning_side"] == 0:
						#top of 9th+ -> bottom of 9th+
						#always happens
						#if home wins from here, it will properly add/retrieve from the file. trust me!
						idx["inning_side"] = 1
						if idx["inning"] > 9 and self.manfred:
							idx["on_2b"] = True
						#i think this should work? like it might be weird if i chunch again but...
					else:
						#bottom of 9th+
						#help
						# score tied -> inning +1
						if idx["away_score"] == idx["home_score"]:
							idx["inning"] += 1
							idx["inning_side"] = 0
							if self.manfred:
								idx["on_2b"] = True
						elif idx["away_score"] > idx["home_score"]:
							idx["outs"] = 3 #ughghghg gross
						else:
							pass #home wins. on walkoffs they will set outs to zero

			new_event = Event(idx["inning"],idx["inning_side"],idx["away_score"],idx["home_score"],
							  idx["outs"],idx["on_1b"],idx["on_2b"],idx["on_3b"], self.home_win, idx["message"], f"{data.MMOLBApiary.URL_NONAPI}watch/{self.game_id}?event={idx["index"]}")
			if new_event != prev_event:
				#print(idx["index"])
				count += 1
				#print(count,new_event, "\n")
				output.append(new_event)
				prev_event = new_event
		return output

class GameWinPercent:
	p = Path(__file__).with_name("winpct.json")
	with open(p, "r") as f:
		winPct = WinPct(json.load(f, object_hook=jsonKeys2int))
	def __init__(self, game_id: str):
		self.ready = False
		self.game_id = game_id
		self.game_json = data.MMOLBApiary.get_game_json(game_id)
		if self.game_json is None:
			print("GWP failed to init game data")
			self.event_list = None
			self.event_list_wpa = None
		else:
			self.event_list = self.game_events()
			self.event_list_wpa = self.game_events_wpa()
			self.ready = True

	def game_events(self) -> list[Event]:
		out = []
		if self.game_json is None:
			#one more try
			self.game_json = data.MMOLBApiary.get_game_json(self.game_id)
			if self.game_json is None:
				print("Still unable to get game data")
				return out
		parser = GameParser(game_id=self.game_id, local_json=self.game_json)
		for event in parser.parse():
			out.append(event)
		return out

	def game_events_wpa(self) -> list[EventWP]:
		out = []
		if len(self.event_list) < 1:
			print(f"No events for game {self.game_id}")
			return out
		for index, event in enumerate(self.event_list):
			win_pct = self.winPct.get_win_pct_event(event)
			if index == 0:
				out.append(EventWP(event, None,win_pct, 0))
				continue
			else:
				prev_event = out[index - 1]
				win_pct_added = win_pct - prev_event.wp
				out.append(EventWP(event, prev_event, win_pct, win_pct_added))
		return out

	def game_events_wpa_sorted(self, home: bool = True):
		if len(self.event_list_wpa) < 1:
			print(f"No events for game {self.game_id}")
			return []
		return sorted(self.event_list_wpa, key= lambda event: event.wpa, reverse=home)
	def game_events_wpa_sorted_abs(self) -> list[EventWP]:
		if len(self.event_list_wpa) < 1:
			print(f"No events for game {self.game_id}")
			return []
		return sorted(self.event_list_wpa, key= lambda event: abs(event.wpa), reverse=True)

	#palettes: unedited, +contrast (maybe check if needed?), high_contrast, splatoon, mesmerizer - hidden :3
	def generate_chart(self, color_palette: Palette):
		# colors
		away_color = "#75d5ff"
		home_color = "#941100"
		match color_palette:
			case Palette.UNEDITED:
				away_color = "#" + self.game_json["AwayTeamColor"]
				home_color = "#" + self.game_json["HomeTeamColor"]
			case Palette.CONTRAST:
				away_color = "#" + self.game_json["AwayTeamColor"]
				home_color = "#" + self.game_json["HomeTeamColor"]
				con = contrast(away_color, home_color)
				away_color = con[0].hex
				home_color = con[1].hex
			case Palette.HIGH_CONTRAST:
				away_color = "#025196"
				home_color = "#fdb138"
			case Palette.SPLATOON:
				pal = random.choice(splatoon_colors)
				away_color = pal[0]
				home_color = pal[1]
			case Palette.MIKUTETO:
				away_color = "#23ccd0"
				home_color = "#a62b39"

		### default values
		arr = []
		xticks = [0]
		xtickslabels = ["t1"]
		yticks = [0, 50, 100]
		#ytickslabels = [self.game_json["AwayTeamEmoji"], "50", self.game_json["HomeTeamEmoji"]]
		ytickslabels = [self.game_json["HomeTeamEmoji"][0], "50", self.game_json["AwayTeamEmoji"][0]]
		inning_half = False
		inning = 1

		for index, eventwp in enumerate(self.event_list_wpa):
			if eventwp.event.homeatbat != inning_half:
				count = 0
				if eventwp.event.inning > inning:
					inning = eventwp.event.inning
				xticks.append(index)
				xtickslabels.append(f"{"b" if eventwp.event.homeatbat else "t"}{eventwp.event.inning}")
				inning_half = eventwp.event.homeatbat
			arr.append(eventwp.wp*100)

		plt.figure(figsize=[inning, 5.56], dpi=125)
		plt.grid(axis="y", c="black", dashes=(3, 2))

		plt.xlim(0, len(arr) - 1)
		plt.ylim(0, 100)
		plt.xticks(xticks, xtickslabels, fontsize=14)
		plt.yticks(yticks, ytickslabels, family=['Segoe UI Emoji', 'DejaVu Sans'], fontsize=20)

		plt.fill_between(range(len(arr)), 100, arr, color=away_color, aa=True)
		plt.fill_between(range(len(arr)), 0, arr, color=home_color, aa=True)

		plt.tight_layout()
		buf = io.BytesIO()
		plt.savefig(buf, format="png")
		plt.close("all")
		buf.seek(0)
		return buf

	#make: chart + big plays
	def generate_embeds(self) -> list[discord.Embed] | None:
		if self.event_list is None or self.game_json is None:
			print(f"No events for game {self.game_id}")
			return None
		embeds = []
		last_event = self.game_json["EventLog"][-1]
		away_score = last_event["away_score"]
		home_score = last_event["home_score"]
		color = self.game_json["AwayTeamColor"] if away_score > home_score else self.game_json["HomeTeamColor"]
		cht_embed = discord.Embed(title=f"{self.game_json["AwayTeamEmoji"]} {self.game_json["AwayTeamName"]}: {away_score}\n"
										f"{self.game_json["HomeTeamEmoji"]} {self.game_json["HomeTeamName"]}: {home_score}")
		cht_embed.set_image(url="attachment://cool.png")
		#cht_embed.set_thumbnail(url="attachment://cool.png")
		cht_embed.set_author(name=f"Season {self.game_json["Season"]} Day {self.game_json["Day"]}, {self.game_json["AwayTeamAbbreviation"]} @ {self.game_json["HomeTeamAbbreviation"]}",
							 url=f"{data.MMOLBApiary.URL_NONAPI}watch/{self.game_id}")
		embeds.append(cht_embed)

		#lets get the top 3 events
		sorted = self.game_events_wpa_sorted_abs()

		for i in sorted[:3]:

			auth = i.pretty_prev_event(self.game_json["AwayTeamAbbreviation"], self.game_json["HomeTeamAbbreviation"])
			icon_url = i.pretty_runners()
			title = i.pretty_message()
			home_team = i.wpa > 0
			emoji = self.game_json["HomeTeamEmoji"] if home_team else self.game_json["AwayTeamEmoji"]
			wp = (i.wp * 100) if home_team else ((1-i.wp) * 100)
			wpa = (i.wpa * 100) if home_team else (i.wpa * -100)
			content = f"{emoji} Win Probability: {wp:0.2f}% (+{wpa:0.2f}%)"
			color = self.game_json["HomeTeamColor"] if home_team else self.game_json["AwayTeamColor"]
			embed = discord.Embed(title=title, description=content, color=discord.Color.from_str("#"+color))
			embed.set_author(name=auth, url=i.event.url, icon_url=icon_url)
			embeds.append(embed)

		return embeds

	def no_dice(self):
		return discord.Embed(title=f"<:outage:1429602929941418158> Wasn't able to find game: {self.game_id}", color=discord.Color.red())



def contrast(color1, color2):
	return color_contrast.modulate(color1, color2, level=AccessibilityLevel.AA, mode=color_contrast.ModulationMode.BOTH)

def OMNICHUNCH():
	#todo: perhaps get all the games first, put in a set, and then chunch.
	winPct = WinPct()
	realm = data.MMOLBApiary.get_state()
	completed_games = set()
	for league_id in realm['LesserLeagues'][14:]:
		lesser = data.MMOLBApiary.get_lesser_league(league_id)
		now = datetime.now()
		print(f"{now.time()} Now Chunching: {lesser['Emoji']} {lesser['Name']}")
		teamlist = lesser["Teams"]
		for team in teamlist:
			print(f"Team {teamlist.index(team)} of {len(teamlist)}: {data.MMOLBApiary.URL_NONAPI}team/{team}")
			for season in range(1,7):
				gamelist = data.MMOLBApiary.get_team_schedule(team, season)["games"]
				print(f"Season {season} Total Games: {len(gamelist)}")
				if len(gamelist) == 0:
					continue
				for game in gamelist:
					if "game_id" not in game.keys():
						#print("skipped: no id")
						continue
					if game["game_id"] in completed_games:
						#print("skipped: ", game["game_id"])
						continue
					gameparser = GameParser(game["game_id"])
					for event in gameparser.parse():
						winPct.insert_occurrence_event(event)
					completed_games.add(game["game_id"])
				print(f"Completed games: {len(completed_games)}")
		with open(f"chunchinprogress.json", "w") as f:
			json.dump(winPct.table, f, indent=4)
	with open("../bits/15-16.json", "w") as f:
		json.dump(winPct.table, f, indent=4)\

def combine_tables(table1: object, table2: object):
	out_table = {}
	homeAtBatList = [False, True]
	for inning in range(1, 11):
		out_table[inning] = {}
		for homeAtBat in homeAtBatList:
			out_table[inning][homeAtBat] = {}
			for homeLead in range(-7, 8):
				out_table[inning][homeAtBat][homeLead] = {}
				for outs in range(0, 3):
					out_table[inning][homeAtBat][homeLead][outs] = {}
					for runners in range(0, 8):

						table1res = table1[inning][homeAtBat][homeLead][outs][runners]
						table2res = table2[inning][homeAtBat][homeLead][outs][runners]
						out_table[inning][homeAtBat][homeLead][outs][runners] = {"homeWins": table1res["homeWins"] + table2res["homeWins"],
								   "occurrences": table1res["occurrences"] + table2res["occurrences"],
								   "homeWinPct": 0.0}
						out_res = out_table[inning][homeAtBat][homeLead][outs][runners]
						if out_res["occurrences"] > 0:
							out_res["homeWinPct"] = out_res["homeWins"]/out_res["occurrences"]
	out_table["awayWins"] = {
		"homeWins": 0,
		"occurrences": 1,
		"homeWinPct": 0.0
	}
	return out_table

if __name__ == "__main__":
	print(Palette.CONTRAST.value)
	print(Palette("unedited".upper()))
	print(Palette("OBAMA"))
	pass

	# gwp = GameWinPercent("690620b7bc5a28e68e53ac94")
	# gwp.generate_embeds()


	#gwp = GameWinPercent("690caa2d0d411ebcb8db0b38")
	#gwp.game_events_wpa_sorted(True)
	#gameparser = GameParser("690caa2d0d411ebcb8db0b38") #b9 not played
	#gameparser = GameParser("6902647db6db8e5cf5bcfafd") #home loses in extras
	#gameparser = GameParser("6902485ab6db8e5cf5bcf3c2")
	#gameparser = GameParser("69025624b6db8e5cf5bcf7c4") #walkoff
	#gameparser = GameParser('68b36654c7846060e4d411bb') #20 innings, no manfred

	#for event in gameparser.parse():
	#	print(f"{event.inning}, {event.homeatbat}, {event.homelead}, {event.outs}, {event.runners:03b}, {winPct.get_win_pct_event(event)*100:0.2f}, "
	#		  f"{event.message}")